//
//  BMKMapAnimation.h
//  MapComponent
//
//  Created by zhaoxiangru on 2024/3/12.
//  Copyright © 2024 Baidu. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <CoreLocation/CLLocation.h>
#import "BMKInterpolator.h"
#import "BMKPolyline.h"

NS_ASSUME_NONNULL_BEGIN
@class BMKMapAnimation;
typedef enum : NSUInteger {
    /**
     * Indicates whether the animation transformation should be applied before the animation starts.
     */
    BMKAnimationFillBefore = 0,
    /**
     * Indicates whether the animation transformation should be applied the first frame of animation starts.
     */
    BMKAnimationFillFirst  = 1,
    /**
     * Indicates whether the animation transformation should be applied after the animation ends.
     */
    BMKAnimationFillAfter  = 2, // @Default
} BMKAnimationFillMode;

typedef enum : NSInteger {
    /**
     * When the animation reaches the end and the repeat count is INFINTE_REPEAT
     * or a positive value, the animation restarts from the beginning.
     * @Default
     */
    BMKAnimationRepeatRestart  = 1,
    /**
     * When the animation reaches the end and the repeat count is INFINTE_REPEAT
     * or a positive value, the animation plays backward (and then forward again).
     */
    BMKAnimationReleatReserse  = 2,
} BMKAnimationRepeatMode;
/**
 * 动画运行的状态.
 */
typedef enum : NSUInteger {
    BMKAnimationStatusStart  = 1,   /** 动画开始运行 */
    BMKAnimationStatusEnd    = 2,   /** 动画结束一次运行 */
    BMKAnimationRepeat       = 3,   /** 动画开始重复运行 */
} BMKAnimationStatus;

/**
 * 动画组添加动画类型.
 */
typedef enum : NSUInteger {
    BMKAnimationSetOrderTypeWith  = 0,   /** 所有动画同时执行 */
    BMKAnimationSetOrderTypeThen    = 1,   /** 所有动画按顺序执行 */
} BMKAnimationSetOrderType;


/// 动画状态回调
@protocol BMKMapAnimationDelegate <NSObject>

- (void)animation:(BMKMapAnimation *)animation status:(BMKAnimationStatus)st;

@end

@interface BMKMapAnimation : NSObject
/// 默认BMKAnimationFillAfter
@property(nonatomic, assign) BMKAnimationFillMode   fillMode;
/// 默认BMKAnimationRepeatRestart
@property(nonatomic, assign) BMKAnimationRepeatMode repeatMode;
@property (nonatomic, strong) NSString *extParam;       // 给端上做透传的参数，引擎不处理

/// When this Animation should start, in milliseconds from the start time of the root AnimationSet
@property(nonatomic, assign) NSInteger startDelay;
@property(nonatomic, assign) NSInteger startTime;
@property(nonatomic, assign) NSInteger duration;
/// 动画重复次数，不包含默认的一次，默认值为0，动画仅开始执行一次后立即结束；设置为1时，动画开始执行1次后会再重复1次，共2次；设置为负数时，动画将一直执行
@property(nonatomic, assign) NSInteger repeatCount;
@property(nonatomic, weak)   id<BMKMapAnimationDelegate> delegate;
@property(nonatomic, strong) BMKInterpolator *interpolator;

@property(nonatomic, readonly) BOOL isStarted;
@property(nonatomic, readonly) BOOL isComplete;

- (BOOL)reset;

- (BOOL)pause;

- (BOOL)resume;

- (BOOL)cancel;

- (BOOL)start;
@end

@interface BMKMapAlphaAnimation : BMKMapAnimation
@property(nonatomic, readonly) float fromAlpha;
@property(nonatomic, readonly) float toAlpha;

- (instancetype)initWithFromAlpha:(float)from to:(float)to;
@end

@interface BMKMapRotateAnimation : BMKMapAnimation
@property(nonatomic, readonly) float fromDegrees;
@property(nonatomic, readonly) float toDegrees;

- (instancetype)initWithFromDegrees:(float)from to:(float)to;
@end

@interface BMKMapScaleAnimation : BMKMapAnimation
/// from缩放比例，相对于原尺寸*比例
@property(nonatomic, readonly) CGSize from;
/// from缩放比例，相对于原尺寸*比例
@property(nonatomic, readonly) CGSize to;

- (instancetype)initWithFromSizeScale:(CGSize)fromSizeScale to:(CGSize)toSizeScale;

@end

@interface BMKMapTranslateAnimation : BMKMapAnimation
@property(nonatomic, readonly) CLLocationCoordinate2D from;
@property(nonatomic, readonly) CLLocationCoordinate2D to;

- (instancetype)initWithFromPoint:(CLLocationCoordinate2D)from to:(CLLocationCoordinate2D)to;
@end

@class BMKPolyline;
@class BMKMapTrackAnimation;
/// 动画状态回调
@protocol BMKMapTrackAnimationDelegate <NSObject>

- (void)animation:(nullable BMKMapTrackAnimation *)animation position:(CLLocationCoordinate2D)pt angle:(float)angle fraction:(float)fraction;

@end
@interface BMKMapTrackAnimation : BMKMapAnimation       // 轨迹动画
@property(nonatomic, weak) id<BMKMapTrackAnimationDelegate> trackDelegate;
/// 轨迹动画的起始和终止比率，范围0.0~1.0， 默认0.0
@property (nonatomic, readonly) CGFloat fromTrackPosRadio;
@property (nonatomic, readonly) CGFloat toTrackPosRadio;
/// 轨迹动画的起始和终止坐标，坐标类型同全局坐标系，内部自动绑路计算动画进度比率
@property (nonatomic, readonly) CLLocationCoordinate2D fromTrackPos;
@property (nonatomic, readonly) CLLocationCoordinate2D toTrackPos;

/// 设置轨迹动画的起始和终止比率
- (BOOL)setTrackPosRadio:(float)from to:(float)to;
/// 设置轨迹动画的终止比率,起始值是当前位置的比率
- (BOOL)setTrackPosRadio:(float)to;
/// 设置轨迹动画的起始和终止坐标
- (BOOL)setTrackPos:(CLLocationCoordinate2D)from to:(CLLocationCoordinate2D)to;
/// 设置轨迹动画的终止坐标，起始值是当前位置的坐标
- (BOOL)setTrackPos:(CLLocationCoordinate2D)to;
/// 设置轨迹
- (BOOL)setTrackLine:(BMKPolyline *)trackLine;
@end


@interface BMKMapAnimationSet : BMKMapAnimation
// 添加动画组成员,orderType为动画执行顺序类型，注意：当第一个add的类型是BMKAnimationSetOrderTypeThen时等同于是BMKAnimationSetOrderTypeWith
- (void)addAnimation:(BMKMapAnimation *)animation addAnimationSetOrderType:(BMKAnimationSetOrderType)orderType;
@end

NS_ASSUME_NONNULL_END
