//
//  BMKBackgroundNavigationCALayer.h
//  MapComponent
//
//  Created by zhaoxiangru on 2024/2/29.
//  Copyright © 2024 Baidu. All rights reserved.
//

#import <QuartzCore/QuartzCore.h>
#import <CoreLocation/CLLocation.h>
#import "BMKMapStatus.h"

@class BMKMapView;

NS_ASSUME_NONNULL_BEGIN
//已走过路线展示方式
typedef NS_ENUM (NSInteger, BMKRoutePassedDisplayMode) {
    BMKRoutePassedDisplayModeNone,           // 无效果
    BMKRoutePassedDisplayModeErased,         //擦除
    BMKRoutePassedDisplayModeGrayed  //置灰
};

//已走过途径点展示方式
typedef NS_ENUM (NSInteger, BMKViaPointPassedDisplayMode) {
    BMKViaPointPassedDisplayModeNone,           // 无效果
    BMKViaPointPassedDisplayModeErased,         //擦除
    BMKViaPointPassedDisplayModeGrayed  //置灰
};

/// 导航（小车）当前位置信息
@interface BMKNaviLocationInfo : NSObject
/// 当前位置经纬度
@property (nonatomic, assign) CLLocationCoordinate2D location;
/// 当前绑路后位置经纬度
@property (nonatomic, assign) CLLocationCoordinate2D locationBindRoute;
/// 当前位置方向
@property (nonatomic, assign) CGFloat direction;
/// 当前位置路线方向
@property (nonatomic, assign) CGFloat directionRoute;
/// 当前位置的路线形状点索引，对应BMKNaviRouteInfo的routeInfo数组
@property (nonatomic, assign) NSInteger shapeIndex;
/// 当前是否偏航（远离路线）
@property (nonatomic, assign) BOOL isFaraway;
@end

/// 路线信息
@interface BMKNaviRouteInfo : NSObject
/// 路线信息
@property (nonatomic, strong, nullable) NSArray<NSValue *> *routeInfo;
/// 路线匹配的路况信息
@property (nonatomic, strong, nullable) NSArray<NSNumber *> *trafficInfo;
@end

/// 途径点信息
@interface BMKNaviViaPointInfo : NSObject
/// 途径点位置信息
@property (nonatomic, assign) CLLocationCoordinate2D location;
/// 途径点是否已走过
@property (nonatomic, assign) BOOL isPassed;
@end

@protocol BMKBackgroundNavigationViewDataSource <NSObject>
// 用于获取导航中用户位置实时信息
- (BMKNaviLocationInfo *)realTimeInfo;

@end
@interface BMKBackgroundNavigationView : UIView
/// BMKBackgroundNavigationView的dataSource
@property (nonatomic, weak) id<BMKBackgroundNavigationViewDataSource> dataSource;
/// 关联源地图，用于同步数据
@property (nonatomic, weak) BMKMapView *mapview;

/// 是否支持绘制引导线（当前位置偏离路线时），默认NO
@property (nonatomic, assign) BOOL needDrawGuideLine;
/// 是否支持绘制引导线（当前位置偏离路线时），默认蓝色虚线
@property (nonatomic, strong) UIColor *guideLineColor;
/// 是否支持绘制引导线（当前位置偏离路线时），默认2point，范围1～30point
@property (nonatomic, assign) CGFloat guideLineWidth;

/// 设置是否绘制起点图标
@property (nonatomic, assign) BOOL needDrawStartPoint;
/// 设置起点经纬度位置
@property (nonatomic, assign) CLLocationCoordinate2D startPoint;
/// 设置起点图标
@property (nonatomic, strong, nullable) UIImage *startPointIcon;
/// 设置起点图标尺寸，默认图片尺寸
@property (nonatomic, assign) CGSize startPointIconSize;
/// 设置起点图标偏移量，默认图片底部中心在经纬度位置（0.0，-0.5），X轴负数向左，Y轴负数向上片，范围值-0.5～0.5
@property (nonatomic, assign) CGPoint startPointIconOffset;

/// 设置是否绘制途径点图标
@property (nonatomic, assign) BOOL needDrawViaPoint;
/// 设置已走过途径点绘制模式
@property (nonatomic, assign) BMKViaPointPassedDisplayMode viaPointPassedDisplayMode;
/// 设置途径点经纬度位置列表
@property (nonatomic, copy, nullable) NSArray<BMKNaviViaPointInfo *> *viaPoints;
/// 设置途径点图标
@property (nonatomic, strong, nullable) UIImage *viaPointIcon;
/// 设置已走过途径点图标
@property (nonatomic, strong, nullable) UIImage *viaPassedPointIcon;
/// 设置途径点图标尺寸，默认图片尺寸
@property (nonatomic, assign) CGSize viaPointIconSize;
/// 设置途径点图标偏移量，默认图片底部中心在经纬度位置（0.0，-0.5），X轴负数向左，Y轴负数向上片，范围值-0.5～0.5
@property (nonatomic, assign) CGPoint viaPointIconOffset;

/// 设置是否绘制终点图标
@property (nonatomic, assign) BOOL needDrawEndPoint;
/// 设置终点经纬度位置
@property (nonatomic, assign) CLLocationCoordinate2D endPoint;
/// 设置终点图标
@property (nonatomic, strong, nullable) UIImage *endPointIcon;
/// 设置终点图标尺寸，默认图片尺寸
@property (nonatomic, assign) CGSize endPointIconSize;
/// 设置终点图标偏移量，默认图片底部中心在经纬度位置（0.0，-0.5），X轴负数向左，Y轴负数向上片，范围值-0.5～0.5
@property (nonatomic, assign) CGPoint endPointIconOffset;

/// 设置导航（小车）图标
@property (nonatomic, strong, nullable) UIImage *navigationIcon;
/// 设置导航（小车）图标尺寸，默认图片尺寸
@property (nonatomic, assign) CGSize navigationIconSize;
/// 是否展示用户实时方向,设置后会从dataSource获取实时数据
@property (nonatomic, assign) BOOL needFllowPhoneHeading;

/// 是否支持绘制罗盘，默认NO
@property (nonatomic, assign) BOOL needDrawCompass;
/// 设置罗盘图标
@property (nonatomic, strong, nullable) UIImage *compassIcon;
/// 设置罗盘图标尺寸，默认图片尺寸
@property (nonatomic, assign) CGSize compassIconSize;

/// 设置路线颜色，默认：#50D27D
@property (nonatomic, strong) UIColor *routeColor;
/// 设置路线线宽，默认：10point，范围1～30point
@property (nonatomic, assign) CGFloat lineWidth;
/// 走过路线绘制模式。默认BMKRoutePassedDisplayModeNone，无任何效果
@property (nonatomic, assign) BMKRoutePassedDisplayMode routePassedDisplayMode;
/// 设置经过的路线颜色
@property (nonatomic, strong) UIColor *routePassedColor;
/// 设置路线路况颜色数组,count必须为5，默认按顺序为未知:#0000FF 畅通：#50D27D 缓行：#FFD046 拥挤：#E72110 严重拥堵：#B4170A 
@property (nonatomic, strong) NSArray<UIColor *> *trafficColor;

/// 更新路线信息
- (void)updateRouteInfo:(BMKNaviRouteInfo *)routeInfo;
/// 更新导航小车绑路后实时位置，方向及绑路后路线形状点索引
- (void)updateNaviLocationInfo:(BMKNaviLocationInfo *)locationInfo;

/// 渲染方法,缩放时起终点、途径点导航图标和罗盘保持原尺寸，路线，引导线保持原宽度
- (void)renderWithMapStatus:(BMKMapStatus *)mapstatus enable3D:(BOOL)enable3D fitScale:(CGFloat)fitScale mapScreenExtOffset:(CGPoint)mapScreenExtOffset;
@end

NS_ASSUME_NONNULL_END
